<?php
namespace AlGhazal\Core;

use AlGhazal\Handlers\AccidentReportHandler;
use AlGhazal\Handlers\CarRentalHandler;
use AlGhazal\Handlers\LimoHandler;
use AlGhazal\Handlers\SessionHandler;
use AlGhazal\Services\WhatsAppService;
use AlGhazal\Utils\Helpers;
use AlGhazal\Utils\MessageFormatter;
use AlGhazal\Core\SessionManager;
use AlGhazal\Handlers\WelcomeHandler;
use AlGhazal\Handlers\MainMenuHandler;
use AlGhazal\Handlers\CustomerServiceHandler;
use AlGhazal\Utils\MenuHelper;

/**
 * Core class for processing incoming messages
 */
class MessageProcessor {
    private ?WhatsAppService $whatsAppService;
    private SessionManager $sessionManager;
    private $handlers = [];
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->whatsAppService = WhatsAppService::getInstance();
        $this->sessionManager = new SessionManager();
        
        $this->registerDefaultHandlers();
    }
    
    /**
     * Register default message handlers
     * 
     * @return void
     */
    private function registerDefaultHandlers() {
        $this->registerHandler(new SessionHandler());
        $this->registerHandler(new WelcomeHandler());
        $this->registerHandler(new MainMenuHandler());
        $this->registerHandler(new CarRentalHandler());
        $this->registerHandler(new LimoHandler());
        $this->registerHandler(new CustomerServiceHandler());
    }

    public function registerHandler($handler) {
        $this->handlers[] = $handler;
    }

    public function processMessage($from, $to, $body, $messageId) {
        // Convert Arabic numerals to English
        $body = Helpers::englishConverter($body);
        
        // Get language from session
        $language = $this->sessionManager->getLanguage($from);
        $state = $this->sessionManager->getState($from);

        if ($this->sessionManager->hasTimedOut($from)) {
            $this->sessionManager->resetSession($from);
        }

        if ($state == 'wait_for_support_response'){
            return;
        }

        // Try each handler until one handles the message
        foreach ($this->handlers as $handler) {
            if ($handler->handle($from, $body, $language)) {
                return;
            }
        }
        
        // If no handler processed the message, send a default response
        $this->sendDefaultResponse($from, $language);
    }

    private function sendDefaultResponse($to, $language) {
        if ($language == 'AR') {
            $message = "عذراً، هذه الخدمة غير متوفرة حالياً. يرجى اختيار خدمة أخرى من القائمة الرئيسية أو التواصل مع خدمة العملاء.";
        } else {
            $message = "Sorry, this service is currently unavailable. Please select another service from the main menu or contact customer service.";
        }
        
        $this->whatsAppService->sendMessage($to, $message);
        
        // Set the session state to 'main_menu' and send the main menu
        $this->sessionManager->setState($to, 'main_menu');
        $this->sendMainMenu($to, $language);
    }

    private function sendMainMenu($to, $language) {
        $message = MenuHelper::getMainMenuMessage($language);
        $this->whatsAppService->sendMessage($to, $message);
    }
}
