<?php
namespace AlGhazal\Core;

/**
 * Class for managing user sessions
 */
class SessionManager {
    private $sessions = [];
    private $sessionFile;
    private $sessionTimeout = 120; // 10 minutes in seconds
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->sessionFile = __DIR__ . '/../../data/sessions.json';
        $this->loadSessions();
    }
    
    /**
     * Load sessions from file
     * 
     * @return void
     */
    private function loadSessions() {
        if (file_exists($this->sessionFile)) {
            $data = file_get_contents($this->sessionFile);
            $this->sessions = json_decode($data, true) ?: [];
        }
    }
    
    /**
     * Save sessions to file
     * 
     * @return void
     */
    private function saveSessions() {
        // Create directory if it doesn't exist
        $dir = dirname($this->sessionFile);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        
        file_put_contents($this->sessionFile, json_encode($this->sessions));
    }
    
    /**
     * Get session data for a user
     * 
     * @param string $userId User ID
     * @return array Session data
     */
    public function getSession($userId) {
        if (isset($this->sessions[$userId])) {
            $session = $this->sessions[$userId];
            
            // Check if session has timed out (10 minutes)
            if (time() - $session['last_activity'] > $this->sessionTimeout) {
                $session['state'] = 'welcome';
                $this->setSession($userId, $session);
            }
            
            return $session;
        }
        
        // Default session for new users
        return [
            'language' => 'AR',
            'state' => 'welcome',
            'data' => [],
            'last_activity' => time()
        ];
    }
    
    /**
     * Set session data for a user
     * 
     * @param string $userId User ID
     * @param array $data Session data
     * @return void
     */
    public function setSession($userId, $data) {
        $this->sessions[$userId] = $data;
        $this->sessions[$userId]['last_activity'] = time();
        $this->saveSessions();
    }
    
    /**
     * Update a specific session field
     * 
     * @param string $userId User ID
     * @param string $key Session key
     * @param mixed $value Session value
     * @return void
     */
    public function updateSession($userId, $key, $value) {
        $session = $this->getSession($userId);
        $session[$key] = $value;
        $this->setSession($userId, $session);
    }
    
    /**
     * Set the language for a user
     * 
     * @param string $userId User ID
     * @param string $language Language code (AR or EN)
     * @return void
     */
    public function setLanguage($userId, $language) {
        $session = $this->getSession($userId);
        $session['language'] = $language;
        $this->setSession($userId, $session);
    }
    
    /**
     * Get the language for a user
     * 
     * @param string $userId User ID
     * @return string Language code (AR or EN)
     */
    public function getLanguage($userId) {
        $session = $this->getSession($userId);
        return $session['language'] ?? 'AR';
    }
    
    /**
     * Set the conversation state for a user
     * 
     * @param string $userId User ID
     * @param string $state Conversation state
     * @return void
     */
    public function setState($userId, $state) {
        $session = $this->getSession($userId);
        $session['state'] = $state;
        $this->setSession($userId, $session);
    }
    
    /**
     * Get the conversation state for a user
     * 
     * @param string $userId User ID
     * @return string Conversation state
     */
    public function getState($userId) {
        $session = $this->getSession($userId);
        return $session['state'] ?? 'welcome';
    }
    
    /**
     * Store data in the user's session
     * 
     * @param string $userId User ID
     * @param string $key Data key
     * @param mixed $value Data value
     * @return void
     */
    public function setData($userId, $key, $value) {
        $session = $this->getSession($userId);
        if (!isset($session['data'])) {
            $session['data'] = [];
        }
        $session['data'][$key] = $value;
        $this->setSession($userId, $session);
    }
    
    /**
     * Get data from the user's session
     * 
     * @param string $userId User ID
     * @param string $key Data key
     * @param mixed $default Default value if key doesn't exist
     * @return mixed Data value
     */
    public function getData($userId, $key, $default = null) {
        $session = $this->getSession($userId);
        return $session['data'][$key] ?? $default;
    }

    /**
     * Check if session has timed out
     * 
     * @param string $userId User ID
     * @return bool Whether session has timed out
     */
    public function hasTimedOut($userId) {
        if (!isset($this->sessions[$userId])) {
            return true;
        }
        
        $session = $this->sessions[$userId];
        return (time() - $session['last_activity']) > $this->sessionTimeout;
    }
    
    /**
     * Clean up expired sessions
     * 
     * @param int $expiryTime Session expiry time in seconds (default: 1 hour)
     * @return void
     */
    public function cleanupSessions($expiryTime = 3600) {
        $now = time();
        foreach ($this->sessions as $userId => $session) {
            if (isset($session['last_activity']) && ($now - $session['last_activity']) > $expiryTime) {
                unset($this->sessions[$userId]);
            }
        }
        $this->saveSessions();
    }

    /**
     * Clear the session for a user
     * 
     * @param string $userId User ID
     * @return void
     */
    public function clearSession($userId) {
        if (isset($this->sessions[$userId])) {
            unset($this->sessions[$userId]);
            $this->saveSessions();
        }
    }
    
    /**
     * Reset a user's session to start fresh
     * 
     * @param string $userId User ID
     * @return bool Success status
     */
    public function resetSession($userId) {
        // Keep the language preference but reset everything else
        $language = $this->getLanguage($userId);
        
        // Create a fresh session with only language preserved
        $sessionData = [
            'language' => $language ?? 'AR',
            'state' => 'welcome',
            'data' => [],
            'last_activity' => time()
        ];
        
        $this->sessions[$userId] = $sessionData;
        $this->saveSessions();
        return true;
    }
}
