<?php
namespace AlGhazal\Handlers;

use AlGhazal\Core\SessionManager;
use AlGhazal\Handlers\CustomerServiceHandler;
use AlGhazal\Utils\MessageFormatter;

/**
 * Handler for main menu options
 */
class MainMenuHandler extends BaseHandler {
    private $sessionManager;
    private $messageFormatter;
    
    /**
     * Constructor
     */
    /**
     * Initialize a new instance of the MainMenuHandler.
     * Sets up the session manager and message formatter.
     */
    public function __construct() {
        parent::__construct();
        $this->sessionManager = new SessionManager();
        $this->messageFormatter = new MessageFormatter();
    }
    
    /**
     * Handle the incoming message and route to appropriate handler based on the message body.
     *
     * @param string $to The recipient's phone number (used as chat ID)
     * @param string $body The message body received
     * @param string $language The language preference (AR/EN)
     * @return bool Returns true if the message was handled, false otherwise
     */
    public function handle($to, $body, $language) {
        $state = $this->sessionManager->getState($to);

        if ($state !== 'main_menu') {
            return false;
        }

        switch ($body) {
            case '1':
                $this->handleCarRental($to, $language);
                return true;

            case '2':
                $this->handleLimo($to, $language);
                return true;

            case '3':
                $this->handleCarRentalReturn($to, $language);
                return true;

            case '4':
                $this->handleLocation($to, $language);
                return true;

            case '0':
                $this->handleCustomerSupport($to, $language);
                return true;

            default:
                $message = ($language == 'AR') 
                    ? "عذراً، الخيار غير صحيح. يرجى اختيار رقم من القائمة."
                    : "Sorry, invalid option. Please select a number from the menu.";
                $this->sendMessage($to, $message);
                $this->sendMainMenu($to, $language);
                return true;
        }
    }

    /**
     * Handle the main menu selection and redirect to welcome handler.
     *
     * @param string $to The recipient's phone number (used as chat ID)
     * @param string $language The language preference (AR/EN)
     */
    private function handleMainMenu($to, $language) {
        $this->sendMainMenu($to, $language);
        return true;
    }

    private function handleCarRental($to, $language) {
        $this->sessionManager->setState($to, 'main_menu');
        (new CarRentalHandler())->handle($to, '1', $language);
    }

    private function handleLimo($to, $language) {
        $this->sessionManager->setState($to, 'main_menu');
        (new LimoHandler())->handle($to, '2', $language);
    }

    private function handleCarRentalReturn($to, $language) {
        $this->sessionManager->setState($to, 'main_menu');
        (new CarRentalHandler())->handle($to, '3', $language);
    }

    private function handleLocation($to, $language) {
        $this->sessionManager->setState($to, 'main_menu');
        (new LocationHandler())->handle($to, '4', $language);
    }

    private function handleCustomerSupport($to, $language){
        $this->sessionManager->setState($to, 'main_menu');
        (new CustomerServiceHandler())->handle($to, '0', $language);
    }
}
