<?php
namespace AlGhazal\Services;

use AlGhazal\Utils\Helpers;

/**
 * Service for interacting with Al Ghazal API
 */
class MaqamApiService {
    private $baseUrl;
    private $endpoints;

    public function __construct() {
        $config = require __DIR__ . '/../Config/config.php';
        $this->baseUrl = $config['ghazal_api']['base_url'];
        $this->endpoints = $config['ghazal_api']['endpoints'];
    }
    
    /**
     * Get car brands
     * 
     * @return array Brands data
     */
    public function getBrands() {
        $url = $this->baseUrl . $this->endpoints['brands'];
        return $this->makeRequest($url);
    }
    
    /**
     * Get brand ID by selection number
     * 
     * @param string $body Selection number
     * @param array $brands List of brands
     * @return int Brand ID
     */
    public function getIDBrand($body, $brands) {
        return $brands[$body-1]['id'];
    }
    
    /**
     * Get contract ID by selection number
     * 
     * @param string $body Selection number (1-based index)
     * @param array $contracts List of contracts
     * @return string|null Contract ID or null if not found
     */
    public function getIDContract($body, $contracts) {
        return $contracts[$body-1]['res_id'] ?? null;
    }
    
    /**
     * Get sub-brands for a specific brand
     * 
     * @param int $idBrand Brand ID
     * @return array Sub-brands data
     */
    public function getSubBrands($idBrand) {
        $url = $this->baseUrl . $this->endpoints['subbrands'] . "?brandId=" . $idBrand;
        return $this->makeRequest($url);
    }
    
    /**
     * Get sub-brand ID by selection number
     * 
     * @param string $body Selection number
     * @param array $subBrands List of sub-brands
     * @return int Sub-brand ID
     */
    public function getIDSubBrand($body, $subBrands) {
        return $subBrands[$body-1]['id'];
    }
    
    /**
     * Get cars for a specific sub-brand
     * 
     * @param int $idSubBrand Sub-brand ID
     * @return array Cars data
     */
    public function getCars($idSubBrand) {
        $url = $this->baseUrl . $this->endpoints['cars'] . "?sub_id=" . $idSubBrand;
        return $this->makeRequest($url);
    }
    
    /**
     * Get contracts/reservations for a mobile number
     * 
     * @param string $mobile Mobile number
     * @return array Contracts data
     */
    public function getContracts($mobile) {
        // Format mobile number as required by the API
        if ($mobile[0] == "9") {
            $mobile = substr($mobile, 3);
            $mobile = "0".$mobile;
        } else {
            $mobile = substr($mobile, 1);
        }
        
        $url = $this->baseUrl . $this->endpoints['contracts'] . "?mobile=" . $mobile;
        return $this->makeRequest($url);
    }
    
    /**
     * Create a ticket
     * 
     * @param string $mobile Mobile number
     * @param string $name Customer name
     * @param int $type Ticket type
     * @param string $message Ticket message
     * @param int $days Days for extension
     * @param int $idReservation Reservation ID
     * @return void
     */
    public function createTicket($mobile, $name, $type, $message, $days, $idReservation) {
        $fields = [
            'mobile' => $mobile,
            'name' => $name,
            'type' => $type,
            'msg' => $message,
            'days' => $days,
            'res' => $idReservation,
        ];
        
        $url = $this->baseUrl . $this->endpoints['ticket'] . '?' . http_build_query($fields);
        $this->makeRequest($url, 'POST', $fields);
    }
    
    /**
     * Update session and get offers
     * 
     * @param string $mobile Mobile number
     * @param int $type Session type
     * @return array Session data with offer information
     */
    public function updateSession($mobile, $type) {
        $params = [
            'mobile' => $mobile,
            'type' => $type
        ];
        
        $url = $this->baseUrl . $this->endpoints['session'] . "?" . http_build_query($params);
        return $this->makeRequest($url);
    }
    
    /**
     * Get promotional offers
     * 
     * @return array Offers data
     */
    public function getOffers() {
        // This method is kept for backward compatibility
        // Use updateSession to get offers information
        $url = $this->baseUrl . $this->endpoints['offers'];
        return $this->makeRequest($url);
    }
    
    /**
     * Get offer details by ID
     * 
     * @param int $offerId Offer ID
     * @return string Offer URL
     */
    public function getOfferUrl($offerId) {
        return "http://admin.alghazal.sa/od/" . $offerId;
    }
    
    /**
     * Get car details URL by ID
     * 
     * @param int $carId Car ID
     * @return string Car details URL
     */
    public function getCarUrl($carId) {
        return "http://admin.alghazal.sa/cd/" . $carId;
    }
    
    /**
     * Create a limousine ticket
     * 
     * @param string $name Customer name
     * @param string $mobile Mobile number
     * @param string $from Start location
     * @param string $to End location
     * @param string $date Journey date
     * @return void
     */
    public function createLimoTicket($name, $mobile, $from, $to, $date) {
        $fields = [
            'mobile' => $mobile,
            'name' => $name,
            'from' => $from,
            'to' => $to,
            'date' => $date,
        ];
        
        $url = $this->baseUrl . $this->endpoints['limo'] . '?' . http_build_query($fields);
        $this->makeRequest($url, 'POST', $fields);
    }
    
    /**
     * Get branch information
     * 
     * @return array Branches data
     */
    public function getBranches() {
        $url = $this->baseUrl . $this->endpoints['branches'];
        return $this->makeRequest($url);
    }
    
    /**
     * Send customer rating
     * 
     * @param string $mobile Mobile number
     * @param int $type Rating type
     * @param int $rate Rating value
     * @param string $comment Rating comment
     * @return void
     */
    public function sendRate($mobile, $type, $rate, $comment) {
        // Format mobile number as required by the API
        if ($mobile[0] == "9") {
            $mobile = substr($mobile, 3);
            $mobile = "0".$mobile;
        } else {
            $mobile = substr($mobile, 1);
        }
        
        $fields = [
            'type' => $type,
            'rate' => $rate,
            'mobile' => $mobile,
            'comment' => $comment,
        ];
        
        $url = $this->baseUrl . $this->endpoints['rating'] . '?' . http_build_query($fields);
        $this->makeRequest($url, 'POST', $fields);
    }
    
    /**
     * Register a new client
     * 
     * @param string $name Client name
     * @param string $phone Phone number
     * @return bool Whether registration was successful
     */
    public function registerClient($name, $phone) {
        // Format phone number if needed
        if ($phone[0] == "9") {
            $phone = substr($phone, 3);
            $phone = "0".$phone;
        } else if ($phone[0] == "+" || (strlen($phone) > 10 && $phone[0] != "0")) {
            $phone = ltrim($phone, "+");
            $phone = substr($phone, -9); // Get last 9 digits
            $phone = "0" . $phone;
        }
        
        // Prepare the registration data
        $data = [
            'name' => $name,
            'mobile' => $phone,
        ];
        
        // Make API call to register the client
        $url = $this->baseUrl . $this->endpoints['register_client'];
        $response = $this->makeRequest($url, 'POST', $data);
        
        // Check if registration was successful
        return isset($response['status']) && $response['status'] === true;
    }

    /**
     * Check if client exists
     * 
     * @param string $mobile Mobile number
     * @return bool Whether client exists
     */
    public function checkClientExist($mobile) {
        // Format mobile number as required by the API
        if ($mobile[0] == "9") {
            $mobile = substr($mobile, 3);
            $mobile = "0".$mobile;
        } else {
            $mobile = substr($mobile, 1);
        }
        
        $url = $this->baseUrl . $this->endpoints['check_client'] . "?mobile=" . $mobile;
        $response = $this->makeRequest($url);
        
        return $response['status'] ?? false;
    }
    
    /**
     * Make an HTTP request to the API
     * 
     * @param string $url API URL
     * @param string $method HTTP method (GET or POST)
     * @param array $fields Request fields for POST
     * @return mixed Response data
     */
    private function makeRequest($url, $method = 'GET', $fields = []) {
        return true;

        $ch = curl_init();
        
        if ($method === 'GET') {
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        } else {
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_ENCODING => "",
                CURLOPT_MAXREDIRS => 10,
                CURLOPT_TIMEOUT => 30,
                CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
                CURLOPT_CUSTOMREQUEST => $method,
                CURLOPT_POSTFIELDS => $fields,
                CURLOPT_HTTPHEADER => [
                    "content-type: application/json"
                ],
            ]);
        }
        
        $response = curl_exec($ch);
        $err = curl_error($ch);
        
        curl_close($ch);
        
        if ($err) {
            return [];
        }
        
        return json_decode($response, true);
    }
}
