<?php
namespace AlGhazal\Services;

use UltraMsg\WhatsAppApi;
use AlGhazal\Utils\Logger;

/**
 * Service for interacting with WhatsApp via UltraMsg SDK
 */
class WhatsAppService {
    private $client;
    private $logger;
    
    /**
     * Static instance for testing purposes
     * @var WhatsAppService
     */
    public static $instance = null;
    
    /**
     * Constructor
     */
    public function __construct() {
        $config = require __DIR__ . '/../Config/config.php';
        $this->client = new WhatsAppApi($config['ultramsg']['token'], $config['ultramsg']['instance']);
        $this->logger = new Logger();
    }
    
    /**
     * Get instance (singleton pattern)
     * 
     * @return WhatsAppService
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Send a text message to a WhatsApp number
     * 
     * @param string $to Recipient phone number
     * @param string $message Message text
     * @return array Response from API
     */
    public function sendMessage($to, $message) {
        // Log the outgoing message
        $this->logger->logResponse($to, $message, 'text');
        
        try {
            return $this->client->sendChatMessage($to . '@c.us', $message);
        } catch (\Exception $e) {
            // Log any errors that occur
            $this->logger->logError("Error sending WhatsApp message", [
                'to' => $to,
                'error' => $e->getMessage()
            ]);
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Send an image with caption to a WhatsApp number
     * 
     * @param string $to Recipient phone number
     * @param string $imageUrl URL of the image
     * @param string $caption Image caption
     * @return array Response from API
     */
    public function sendImage($to, $imageUrl, $caption = '') {
        // Log the outgoing image message
        $this->logger->logResponse($to, $caption, 'image', ['image_url' => $imageUrl]);
        
        try {
            return $this->client->sendImageMessage($to . '@c.us', $imageUrl, $caption);
        } catch (\Exception $e) {
            // Log any errors that occur
            $this->logger->logError("Error sending WhatsApp image", [
                'to' => $to,
                'image_url' => $imageUrl,
                'error' => $e->getMessage()
            ]);
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get messages from a chat
     * 
     * @param string $chatId Chat ID
     * @param int $limit Number of messages to retrieve (default: 10)
     * @param string $status Message status (default: all)
     * @return array Messages
     */
    public function getMessages($chatId, $limit = 10, $status = 'all') {
        return $this->client->getMessages($chatId, $status, $limit);
    }
    
    /**
     * Get contact information
     * 
     * @param string $chatId Chat ID (phone number with @c.us suffix)
     * @return array Contact information including name
     */
    public function getContact($chatId) {
        try {
            // Ensure chatId has @c.us suffix if it's a phone number
            if (is_numeric($chatId) && strpos($chatId, '@') === false) {
                $chatId .= '@c.us';
            }
            
            return $this->client->getContact($chatId);
        } catch (\Exception $e) {
            // Log any errors that occur
            $this->logger->logError("Error getting contact info", [
                'chatId' => $chatId,
                'error' => $e->getMessage()
            ]);
            return ['error' => $e->getMessage()];
        }
    }
    
    /**
     * Get contact name
     * 
     * @param string $chatId Chat ID (phone number with @c.us suffix)
     * @return string|null Contact name or null if not found
     */
    public function getContactName($chatId) {
        $contact = $this->getContact($chatId);
        return $contact['name'] ?? null;
    }
}
