<?php
namespace AlGhazal\Utils;

/**
 * Logger utility for tracking requests and responses
 */
class Logger {
    private $config;
    private $logDir;
    private $requestLogFile;
    private $responseLogFile;
    private $errorLogFile;
    private $enabled;
    private $logLevel;
    private $lastError = '';
    
    /**
     * Log levels in order of severity
     */
    const LOG_LEVELS = [
        'debug' => 0,
        'info' => 1,
        'warning' => 2,
        'error' => 3
    ];
    
    /**
     * Constructor
     */
    public function __construct() {
        try {
            $this->config = require __DIR__ . '/../Config/config.php';
            $this->enabled = $this->config['logging']['enabled'] ?? false;
            $this->logDir = rtrim($this->config['logging']['log_dir'] ?? __DIR__ . '/../../logs', '/');
            $this->requestLogFile = $this->logDir . '/' . ($this->config['logging']['request_log'] ?? 'requests.log');
            $this->responseLogFile = $this->logDir . '/' . ($this->config['logging']['response_log'] ?? 'responses.log');
            $this->errorLogFile = $this->logDir . '/' . ($this->config['logging']['error_log'] ?? 'errors.log');
            $this->logLevel = strtolower($this->config['logging']['log_level'] ?? 'info');
            
            // Create log directory if it doesn't exist
            if ($this->enabled) {
                if (!is_dir($this->logDir) && !@mkdir($this->logDir, 0777, true) && !is_dir($this->logDir)) {
                    $this->lastError = "Failed to create log directory: " . $this->logDir;
                    error_log($this->lastError);
                    $this->enabled = false;
                    return;
                }
                
                // Ensure log files are writable or can be created
                $files = [
                    $this->requestLogFile,
                    $this->responseLogFile,
                    $this->errorLogFile
                ];
                
                foreach ($files as $file) {
                    if (file_exists($file) && !is_writable($file)) {
                        $this->lastError = "Log file is not writable: $file";
                        error_log($this->lastError);
                        $this->enabled = false;
                        return;
                    } elseif (!file_exists($file)) {
                        // Try to create the file
                        if (@file_put_contents($file, '') === false) {
                            $this->lastError = "Failed to create log file: $file";
                            error_log($this->lastError);
                            $this->enabled = false;
                            return;
                        }
                    }
                }
            }
        } catch (\Exception $e) {
            $this->lastError = "Logger initialization error: " . $e->getMessage();
            error_log($this->lastError);
            $this->enabled = false;
        }
    }
    
    /**
     * Get the last error that occurred in the logger
     * 
     * @return string The last error message
     */
    public function getLastError() {
        return $this->lastError;
    }
    
    /**
     * Check if the given log level is enabled
     * 
     * @param string $level Log level to check
     * @return bool Whether the log level is enabled
     */
    private function isLevelEnabled($level) {
        if (!$this->enabled) {
            return false;
        }
        
        $configLevel = self::LOG_LEVELS[$this->logLevel] ?? self::LOG_LEVELS['info'];
        $messageLevel = self::LOG_LEVELS[strtolower($level)] ?? self::LOG_LEVELS['info'];
        
        return $messageLevel >= $configLevel;
    }
    
    /**
     * Format a log message with timestamp and additional context
     * 
     * @param string $message Log message
     * @param array $context Additional context
     * @return string Formatted log message
     */
    private function formatLogMessage($message, array $context = []) {
        $timestamp = date('Y-m-d H:i:s');
        $formattedMessage = "[$timestamp] $message";
        
        if (!empty($context)) {
            $formattedMessage .= ' ' . json_encode($context, JSON_UNESCAPED_UNICODE);
        }
        
        return $formattedMessage . PHP_EOL;
    }
    
    /**
     * Write a message to a log file
     * 
     * @param string $file Log file path
     * @param string $message Log message
     * @param array $context Additional context
     * @return bool True on success, false on failure
     */
    private function writeToLog($file, $message, array $context = []) {
        if (!$this->enabled) {
            return false;
        }
        
        $formattedMessage = $this->formatLogMessage($message, $context);
        
        // Use error suppression to prevent warnings, we'll handle errors manually
        $result = @file_put_contents($file, $formattedMessage, FILE_APPEND | LOCK_EX);
        
        if ($result === false) {
            $error = error_get_last();
            $this->lastError = "Failed to write to log file ($file): " . ($error['message'] ?? 'Unknown error');
            error_log($this->lastError);
            return false;
        }
        
        return true;
    }
    
    /**
     * Log an incoming request
     * 
     * @param string $from Sender phone number
     * @param string $body Message body
     * @param array $event Full event data
     * @return bool True if logged successfully, false otherwise
     */
    public function logRequest($from, $body, array $event = []) {
        if (!$this->isLevelEnabled('info')) {
            return true;
        }
        
        $message = "Incoming request from $from: " . substr($body, 0, 100);
        $context = [
            'timestamp' => time(),
            'from' => $from,
            'body' => $body,
            'event' => $event
        ];
        
        return $this->writeToLog($this->requestLogFile, $message, $context);
    }
    
    /**
     * Log an outgoing response
     * 
     * @param string $to Recipient phone number
     * @param string $message Message content
     * @param string $type Message type (text, image, etc.)
     * @param array $additionalData Additional data
     * @return bool True if logged successfully, false otherwise
     */
    public function logResponse($to, $message, $type = 'text', array $additionalData = []) {
        if (!$this->isLevelEnabled('info')) {
            return true;
        }
        
        $logMessage = "Outgoing response to $to: " . substr($message, 0, 100);
        $context = [
            'timestamp' => time(),
            'to' => $to,
            'message' => $message,
            'type' => $type,
            'data' => $additionalData
        ];
        
        return $this->writeToLog($this->responseLogFile, $logMessage, $context);
    }
    
    /**
     * Log an error
     * 
     * @param string $message Error message
     * @param array $context Additional context
     * @return bool True if logged successfully, false otherwise
     */
    public function logError($message, array $context = []) {
        if (!$this->isLevelEnabled('error')) {
            return true;
        }
        
        return $this->writeToLog($this->errorLogFile, $message, $context);
    }
    
    /**
     * Log a debug message
     * 
     * @param string $message Debug message
     * @param array $context Additional context
     * @return bool True if logged successfully, false otherwise
     */
    public function debug($message, array $context = []) {
        if (!$this->isLevelEnabled('debug')) {
            return true;
        }
        
        return $this->writeToLog($this->requestLogFile, "[DEBUG] $message", $context);
    }
    
    /**
     * Log an info message
     * 
     * @param string $message Info message
     * @param array $context Additional context
     * @return bool True if logged successfully, false otherwise
     */
    public function info($message, array $context = []) {
        if (!$this->isLevelEnabled('info')) {
            return true;
        }
        
        return $this->writeToLog($this->requestLogFile, "[INFO] $message", $context);
    }
    
    /**
     * Log a warning message
     * 
     * @param string $message Warning message
     * @param array $context Additional context
     * @return bool True if logged successfully, false otherwise
     */
    public function warning($message, array $context = []) {
        if (!$this->isLevelEnabled('warning')) {
            return true;
        }
        
        return $this->writeToLog($this->errorLogFile, "[WARNING] $message", $context);
    }
}
