<?php

require_once ('vendor/autoload.php');

// Import required classes
use AlGhazal\Core\MessageProcessor;
use AlGhazal\Utils\Logger;

// Initialize logger
$logger = new Logger();

// Log the raw input for debugging
$rawInput = file_get_contents("php://input");
$logger->debug("Raw webhook input received", ['raw_input' => $rawInput]);

// Parse the JSON data
$event = json_decode($rawInput, true);

// Process the webhook event if it exists
if (json_last_error() !== JSON_ERROR_NONE) {
    $logger->logError("Invalid JSON received in webhook", [
        'error' => json_last_error_msg(),
        'raw_input' => $rawInput
    ]);
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON data']);
    exit;
}

// Log the parsed event
$logger->debug("Parsed webhook event", ['event' => $event]);

// Process the webhook event if it exists
if (!empty($event) && is_array($event)) {
    try {
        // Extract message details with error checking
        $to = $event['data']['from'] ?? null;
        $body = $event['data']['body'] ?? '';
        $idChat = $to;
        
        if (empty($to)) {
            $logger->logError("Missing 'from' field in webhook data", ['event' => $event]);
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Missing required field: from']);
            exit;
        }
        
        // Log the incoming message
        $logger->logRequest($to, $body, $event);
        
        // Format the phone number by removing the @c.us suffix if present
        $formattedTo = str_ends_with($to, '@c.us') ? substr($to, 0, -5) : $to;
        
        // Create a new message processor and process the message
        $processor = new MessageProcessor();
        
        // Log the message processing
        $logger->debug("Processing message", [
            'from' => $to,
            'to' => $formattedTo, 
            'body' => $body
        ]);
        
        try {
            // Process the message - using $to as both from and chat ID, and $formattedTo as the recipient
            $processor->processMessage($to, $formattedTo, $body, $to);
            
            // Log successful processing
            $logger->debug("Message processed successfully", [
                'from' => $to,
                'to' => $formattedTo
            ]);
            
        } catch (\Exception $e) {
            // Log any exceptions during processing
            $logger->logError("Error in processMessage: " . $e->getMessage(), [
                'exception' => get_class($e),
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Send an error response to the user
            $errorMessage = "An error occurred while processing your message. Please try again later.";
            $whatsAppService = new \AlGhazal\Services\WhatsAppService();
            $whatsAppService->sendMessage($formattedTo, $errorMessage);
            
            throw $e; // Re-throw to be caught by the outer try-catch
        }
        
        // Return a success response
        http_response_code(200);
        echo json_encode(['status' => 'success', 'message' => 'Message processed']);
        
    } catch (\Exception $e) {
        // Log any exceptions that occur during processing
        $logger->logError("Error processing webhook: " . $e->getMessage(), [
            'exception' => get_class($e),
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString(),
            'event' => $event
        ]);
        
        // Return an error response
        http_response_code(500);
        echo json_encode([
            'status' => 'error', 
            'message' => 'An error occurred while processing the message',
            'error' => $e->getMessage()
        ]);
    }
} else {
    // Log invalid or empty event data
    $logger->logError("Empty or invalid event data received", ['raw_input' => $rawInput]);
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Empty or invalid event data']);
}
